/*************************************************************

  m a i n . c

  Source file for Wave Runner

 *************************************************************

  Firmware description:

  DSP Kernel that calls a processing function for each sample or for a block of samples
  It can operate in sample mode or in block mode
  In sample mode, the processing function is executed for each sample
  In block mode, the processing function is executed a block of samples

  Configuration is defined in configuration.h

  Operation in sample mode:

     Cycle n   : Sample n   is read : Sample n-1 is processed : Sample n-2 is sent to output
     Cycle n+1 : Sample n+1 is read : Sample n   is processed : Sample n-1 is sent to output
     Cycle n+2 : Sample n+2 is read : Sample n+1 is processed : Sample n   is sent to output

     As sample n is read on cycle n and sent to output on cycle n+2, we have a 2 cycle latency

     A blinking Green LED shows sampling operation blinking at fsample/10000 frequency

     PRO1 line shows CPU usage and is in high state during the processing function operation
     Its frequency shall be equal to the sample frequency

  Operation in block mode:

     The latency is equal to the block size multiplied by the sample period

     A blinking Green LED blinks at each processing block

     PRO1 line shows CPU usage and is in high state during the processing function operation
     Its frequency shall be equal to the block frequency

  The LEDs can also be controlled by the main loop or the processing function

  User Button enables input monitoring regardless of the mode
  The switches can also be read by any other function

  The rest of profiling lines are not used by the kernel

  Startup operation:

  Some switches change the firmware operation if they are pressed during reset:

     SW1 : Disables the input anti aliasing filter clock
     SW2 : Disables the output anti imaging filter clock

 *************************************************************/

#include "arm_math.h"       // Include for the DSP lib
#include "Base.h"           // Basic definitions
#include "configuration.h"  // Firmware configuration
#include "dsp_kernel.h"     // DSP Single Sample Kernel
#include "processing.h"     // Processing code
#include "mainLoop.h"       // Main loop functions
#include "chprintf.h"       // Serial printf


// Some macro gymnastics
// Define 3 macros that expand to function names as strings
#define STR(X) #X
#define ASSTR(X) STR(X)
#define NPROC   ASSTR(PROC_FUNCTION)   // String with sample to sample function name
#define NBLOCK  ASSTR(BLOCK_FUNCTION)  // String with block function name
#define NLOOP   ASSTR(LOOP_FUNCTION)   // String with main look function name

/****** Functions ********************************************/

SerialConfig SConfig;  // Serial configuration

// Initialize the serial communication
static void serialInit(void)
{
palSetPadMode(USART2_TX_PORT,USART2_TX_PAD,PAL_MODE_OUTPUT_PUSHPULL);
palSetPadMode(USART2_RX_PORT,USART2_RX_PAD,PAL_MODE_INPUT);
palSetPadMode(USART2_TX_PORT,USART2_TX_PAD,PAL_MODE_ALTERNATE(7));
palSetPadMode(USART2_RX_PORT,USART2_RX_PAD,PAL_MODE_ALTERNATE(7));

// Serial driver initialization
SConfig.speed=SSPEED;         // Configure speed
sdStart(&SDRIVER,&SConfig);   // Initializes serial driver
}

// Show current firmware configuration
static void showConfiguration(void)
 {
 chprintf(SERIAL," Sample frequency is %d Hz%s",SAMPLE_FREQ,BREAK);

 #ifdef ALIAS_BW1
 chprintf(SERIAL," Anti-alias filter corner is at %d Hz%s",ALIAS_BW1,BREAK);
 #else
 chprintf(SERIAL," Anti-alias filter is not defined%s",BREAK);
 #endif

 #ifdef ALIAS_BW2
 chprintf(SERIAL," Anti-image filter corner is at %d Hz%s",ALIAS_BW2,BREAK);
 #else
 chprintf(SERIAL," Anti-image filter is not defined%s",BREAK);
 #endif

 #ifdef BLOCK_MODE
 chprintf(SERIAL," Processing performed in block mode%s",BREAK);
 chprintf(SERIAL," Block function: %s%s",NBLOCK,BREAK);
 #else
 chprintf(SERIAL," Processing performed in sample to sample mode%s",BREAK);
 chprintf(SERIAL," Processing function: %s%s",NPROC,BREAK);
 #endif

 chprintf(SERIAL," Loop function: %s%s",NLOOP,BREAK);

 chprintf(SERIAL,"%s",BREAK);
 }

#ifdef BLOCK_MODE
// Working area for the processing thread in block mode
static WORKING_AREA(waBlock,128);

// Binary semaphore for block mode
BinarySemaphore   blockSemaf;
#endif

#ifdef BLOCK_MODE
// Function that starts the block processing thread
static msg_t thBlockProcess(void *arg)
 {
 PRO1_CLEAR; // Clear profiling line #1
 while (1)
    {
    // Wait for semaphore synchronization
    chBSemWait(&blockSemaf);

    PRO1_SET; // Set profiling line #1

    // Call the block processing function
    blockProcessingPtr();

    PRO1_CLEAR; // Clear profiling line #1
    }

 // This function never returns
 // Return is included so the compiler does not complain
 return 0;
 }
#endif

// Print a sin/cos table
static void sin_cos_table(void)
 {
 int i;
 float32_t a,s,c;

 // Header for the table
 chprintf(SERIAL,"%sSin/Cos table%s%s",BREAK,BREAK,BREAK);
 chprintf(SERIAL,"  Angle     Sin     Cos%s",BREAK);

 // Angle loop
 for(i=0;i<=360;i++)
    {
	a=i*PI/180.0f;     // Convert from deg. to rad.
	s=arm_sin_f32(a);  // Fast sin library function
	c=arm_cos_f32(a);  // Fast cos library function

	// Print this line of the table
	chprintf(SERIAL,"%7d %7f %7f %s",i,s,c,BREAK);
    }

 }

// Print a sin/cos table
static void sin_cos_table15(void)
 {
 int i;
 q15_t a,s,c;

 // Header for the table
 chprintf(SERIAL,"%sSin/Cos table%s%s",BREAK,BREAK,BREAK);
 chprintf(SERIAL,"  Angle     Sin     Cos%s",BREAK);

 // Angle loop
 for(i=0;i<=360;i+=10)
    {
	a = i*91;
	s=arm_sin_q15(a);  // Fast sin library function
	c=arm_cos_q15(a);  // Fast cos library function

	// Print this line of the table
	chprintf(SERIAL,"%7d %d %d %s",i,s,c,BREAK);
    }

 }

int main(void)
 {
 // Basic initializations
 baseInit();

 // Start the serial communication
 serialInit();

 // Show the firmware version on serial channel
 chprintf(SERIAL,"%s=================================================%s",BREAK,BREAK);
 chprintf(SERIAL,  " DSP Firmware Kernel %s%s",VERSION,BREAK);
 chprintf(SERIAL,  "=================================================%s%s",BREAK,BREAK);

 // Show current configuration
 showConfiguration();

 // Code for block mode -----------------------------------------------------------------
 #ifdef BLOCK_MODE

 // Initializes the semaphore as taken
 chBSemInit(&blockSemaf,TRUE);

 // Block mode processing thread creation
 chThdCreateStatic(waBlock, sizeof(waBlock), NORMALPRIO+2, thBlockProcess, NULL);

 // Initialize the FFT tables
 fftInit();

 #endif
 // End of block mode code -------------------------------------------------------------

 // Start the DSP kernel
 startDSP();

 // Messages before the loop function
 chprintf(SERIAL," Entering low priority main loop%s",BREAK);
 chprintf(SERIAL," System start completed%s",BREAK);
 chprintf(SERIAL,"%s=================================================%s%s",BREAK,BREAK,BREAK);

 //sin_cos_table15();

 // Execute the loop function
 // This function should never returns
 //LOOP_FUNCTION();
 mainLoopPtr();


 // Fall back if the loop function returns
 chprintf(SERIAL,"%sERROR : Return from loop function%s",BREAK,BREAK);
 while (1) {};

 // Return so that the compiler doesn't complain
 // It is not really needed as the loop function never returns
 return 0;
 }

