/*************************************************************

   d s p _ k e r n e l . c

   DSP kernel for single sample processing

   APB2 frequency is 84MHz
   Timer8 clock input frequency is 168MHz

   If saturation is detected, the main loop sat counter
   for the affected channel is set to SAT_COUNT (mainLoop.h)
   It is up to the main loop function to use this information

 *************************************************************/

#include "arm_math.h"       // Include for the DSP lib
#include "base.h"           // Basic MCU configuration
#include "configuration.h"  // Firmware configuration
#include "dsp_kernel.h"     // DSP Kernel
#include "processing.h"     // DSP processing functions
#include "mainLoop.h"       // Main loop functions
#include "chprintf.h"       // Serial printf



/** Configurations *******************************************/

#define PWM_CLOCK     12000000                     // 12MHz PWM clock frequency
#ifdef ALIAS_BW1
#define PWM1_CYCLES  (PWM_CLOCK/(ALIAS_BW1*100))   // Number of cycles for input filter clock
#define PWM1_HALF    (PWM1_CYCLES/2)               // Value for 50% duty
#endif
#ifdef ALIAS_BW2
#define PWM2_CYCLES  (PWM_CLOCK/(ALIAS_BW2*100))   // Number of cycles for output filter clock
#define PWM2_HALF    (PWM2_CYCLES/2)               // Value for 50% duty
#endif

/* Configuration for clock 1 (input) PWM on Timer5 */
#ifdef ALIAS_BW1
static PWMConfig pwmcfg1 = {
  PWM_CLOCK,                                // Counter clock frequency
  PWM1_CYCLES,                              // Counts per cycle
  NULL,                                     // No callback
  {
   {PWM_OUTPUT_DISABLED, NULL},
   {PWM_OUTPUT_ACTIVE_HIGH, NULL},  // CH2 activated without callback
   {PWM_OUTPUT_DISABLED, NULL},
   {PWM_OUTPUT_DISABLED, NULL}
  },
  0,
  0
};
#endif

/* Configuration for clock 2 (output) PWM on Timer2 */
#ifdef ALIAS_BW2
static PWMConfig pwmcfg2 = {
  PWM_CLOCK,                                // Counter clock frequency
  PWM2_CYCLES,                              // Counts per cycle
  NULL,                                     // No callback
  {
   {PWM_OUTPUT_DISABLED, NULL},
   {PWM_OUTPUT_DISABLED, NULL},
   {PWM_OUTPUT_DISABLED, NULL},
   {PWM_OUTPUT_ACTIVE_HIGH, NULL}    // CH4 activated without callback
  },
  0,
  0
};
#endif

/** Variables ************************************************/

// Pointer for the sample mode processing function
void (*sampleProcessingPtr)(q15_t,q15_t,q15_t*,q15_t*) = PROC_FUNCTION;

// Pointer for the block mode processing function
void (*blockProcessingPtr)(void) = BLOCK_FUNCTION;

// Pointer for the main loop function
void (*mainLoopPtr)(void) = LOOP_FUNCTION;

// True if we are in block mode
#ifdef BLOCK_MODE
  int32_t blockMode = 1;
#else
  int32_t blockMode = 0;
#endif

// Block buffers as Q15 types
volatile q15_t inbuff1[2*HBLOCK_SIZE];  // Input buffer for channel 1
volatile q15_t inbuff2[2*HBLOCK_SIZE];  // Input buffer for channel 2
volatile q15_t outbuff1[2*HBLOCK_SIZE]; // Output buffer for channel 1
volatile q15_t outbuff2[2*HBLOCK_SIZE]; // Output buffer for channel 2

// Pointers to Q15 objects for block mode
//volatile q15_t *inp1;  // Pointer for channel 1 input
//volatile q15_t *inp2;  // Pointer for channel 2 input
//volatile q15_t *outp1; // Pointer for channel 1 output
//volatile q15_t *outp2; // Pointer for channel 2 output

// Position in buffer for current block
volatile uint32_t blkStart;

/** Private functions ****************************************/

// Configure the four switches
static void configureSwitches(void)
 {
 // We don't need internal pullups because there are pull-ups on the board
 palSetPadMode(SW1_PORT,SW1_PAD,PAL_MODE_INPUT);
 palSetPadMode(SW2_PORT,SW2_PAD,PAL_MODE_INPUT);
 palSetPadMode(SW3_PORT,SW3_PAD,PAL_MODE_INPUT);
 palSetPadMode(SW4_PORT,SW4_PAD,PAL_MODE_INPUT);
 chprintf(SERIAL,"   Switches configured%s",BREAK);
 }

// Configure Timer8 that drives the ADC and DAC sampling
static void configureADCtimer(void)
 {
 // Start the clock on Timer8
 SET_FLAG(RCC->APB2ENR,RCC_APB2ENR_TIM8EN);

 // Set Prescaler to 1
 TIM8->PSC = 0;

 // Set counter limit for 10kHz
 TIM8->ARR = (TIM8_CLK_FREQ/SAMPLE_FREQ);

 // TRGO is set to update event
 SET_FLAG(TIM8->CR2,BIT5);

 // Program the NVIC
 nvicEnableVector(TIM8_UP_TIM13_IRQn,
 		 CORTEX_PRIORITY_MASK(STM32_GPT_TIM8_IRQ_PRIORITY));

 // Set interrupt on update
 SET_FLAG(TIM8->DIER,TIM_DIER_UIE);

 // Enable the counter
 SET_FLAG(TIM8->CR1,TIM_CR1_CEN);

 chprintf(SERIAL,"   ADC timer configured%s",BREAK);
 }

// Configure Timer2 that drives the filter clocks
static void configureClockTimer(void)
 {
 #ifdef	ALIAS_BW1
 if (!SW1_ACTIVE)
    {
    // Start the driver for input filter
    pwmStart(&CLOCK1_DRV, &pwmcfg1);
    // Configure hardware output for TIM5 CH2 on
    palSetPadMode(CLOCK1_PORT,CLOCK1_PAD, PAL_MODE_ALTERNATE(CLOCK1_AF));
    // Set mode to open drain
    SET_FLAG(CLOCK1_PORT->OTYPER,BIT(CLOCK1_PAD));
    // Set 2MHz speed (Minimum)
    CLEAR_FLAG(CLOCK1_PORT->OSPEEDR,BIT(2*CLOCK1_PAD));
    CLEAR_FLAG(CLOCK1_PORT->OSPEEDR,BIT(2*CLOCK1_PAD+1));
    // Set PWM to 50%
    pwmEnableChannel(&CLOCK1_DRV,CLOCK1_CHANNEL-1,PWM1_HALF);

    chprintf(SERIAL,"   Anti-alias clock started%s",BREAK);
    }
 #endif //ALIAS_BW1

 #ifdef	ALIAS_BW2
 if (!SW2_ACTIVE)
    {
    // Start the driver for output filter
    pwmStart(&CLOCK2_DRV, &pwmcfg2);
    // Configure hardware output for TIM5 CH2 on
    palSetPadMode(CLOCK2_PORT,CLOCK2_PAD, PAL_MODE_ALTERNATE(CLOCK2_AF));
    // Set mode to open drain
    SET_FLAG(CLOCK2_PORT->OTYPER,BIT(CLOCK2_PAD));
    // Set 2MHz speed (Minimum)
    CLEAR_FLAG(CLOCK2_PORT->OSPEEDR,BIT(2*CLOCK2_PAD));
    CLEAR_FLAG(CLOCK2_PORT->OSPEEDR,BIT(2*CLOCK2_PAD+1));
    // Set PWM to 50%
    pwmEnableChannel(&CLOCK2_DRV,CLOCK2_CHANNEL-1,PWM2_HALF);

    chprintf(SERIAL,"   Anti-image clock started%s",BREAK);
    }
 #endif //ALIAS_BW2
 }

// Configure the DAC outputs
static void configureDAC(void)
 {
 // Set pin outputs
 palSetPadMode(DAC_PORT,DAC1_PAD,PAL_MODE_INPUT_ANALOG);
 palSetPadMode(DAC_PORT,DAC2_PAD,PAL_MODE_INPUT_ANALOG);

 // Start the clock on DAC peripheral
 SET_FLAG(RCC->APB1ENR,RCC_APB1ENR_DACEN);

 // Set DAC1 and DAC2 update on Timer8 TRGO
 SET_FLAG(DAC->CR,BIT19|BIT18|BIT3|BIT2);

 // Enable DAC1 and DAC2
 SET_FLAG(DAC->CR,DAC_CR_EN1|DAC_CR_EN2);

 chprintf(SERIAL,"   DACs configured%s",BREAK);
 }

// Configure the ADC inputs
static void configureADC(void)
 {
 // Enable all three ADC clocks
 SET_FLAG(RCC->APB2ENR
		 ,RCC_APB2ENR_ADC1EN|RCC_APB2ENR_ADC2EN|RCC_APB2ENR_ADC3EN);

 // ADC pin configurations
 palSetPadMode(ADC1_PORT,ADC1_PAD,PAL_MODE_INPUT_ANALOG);
 palSetPadMode(ADC2_PORT,ADC2_PAD,PAL_MODE_INPUT_ANALOG);
 // Potentiometer configurations
 palSetPadMode(POT1_PORT,POT1_PAD,PAL_MODE_INPUT_ANALOG);
 palSetPadMode(POT2_PORT,POT2_PAD,PAL_MODE_INPUT_ANALOG);

 // Set ADC sample times
 // This configuration does fully use the header file information
 // Configuration for ADC1 and ADC2
 BIT_FIELD_WRITE(ADC1->SMPR1,3*(ADC1_IN-10),7,7);   // For inputs >9
 BIT_FIELD_WRITE(ADC2->SMPR2,3*ADC2_IN     ,7,7);   // For inputs <10
 // Configuration for potentiometers
 BIT_FIELD_WRITE(ADC3->SMPR1,3*(POT1_IN-10),7,7);   // For inputs >9
 BIT_FIELD_WRITE(ADC3->SMPR1,3*(POT2_IN-10),7,7);   // For inputs >9

 // Configure ADC1,2,3 for just one conversion
 BIT_FIELD_WRITE(ADC1->SQR1,20,15,0);
 BIT_FIELD_WRITE(ADC2->SQR1,20,15,0);
 BIT_FIELD_WRITE(ADC3->SQR1,20,15,0);

 // Set channel to use on ADC1.2
 // ADC3 is not set as it changes when reading potentiometers
 BIT_FIELD_WRITE(ADC1->SQR3,0,31,ADC1_IN);
 BIT_FIELD_WRITE(ADC2->SQR3,0,31,ADC2_IN);

 // Set Timer8 TRGO Triggering for ADC1,2
 SET_FLAG(ADC1->CR2,BIT28|BIT27|BIT26|BIT25);
 SET_FLAG(ADC2->CR2,BIT28|BIT27|BIT26|BIT25);

 // Activate all ADCs
 ADC1->CR2 |= ADC_CR2_ADON;
 ADC2->CR2 |= ADC_CR2_ADON;
 ADC3->CR2 |= ADC_CR2_ADON;

 chprintf(SERIAL,"   ADCs configured%s",BREAK);
 }

// Initialize profiling pins
static void startProfiling(void)
 {
 palSetPadMode(PRO1_PORT,PRO1_PAD,PAL_MODE_OUTPUT_PUSHPULL);
 palSetPadMode(PRO2_PORT,PRO2_PAD,PAL_MODE_OUTPUT_PUSHPULL);
 palSetPadMode(PRO3_PORT,PRO3_PAD,PAL_MODE_OUTPUT_PUSHPULL);
 palSetPadMode(PRO4_PORT,PRO4_PAD,PAL_MODE_OUTPUT_PUSHPULL);

 chprintf(SERIAL,"   Profiling lines configured%s",BREAK);
 }

/** Public Functions *****************************************/

// Initialize and run the DSP kernel
void startDSP(void)
 {
 chprintf(SERIAL," Starting DSP Kernel%s",BREAK);

 // Initialize profiling lines
 startProfiling();

 // Configure Switches
 configureSwitches();

 // Configure ADC inputs
 configureADC();

 // Configure the DAC outputs
 configureDAC();

 // Configure and start the filter clock timer
 configureClockTimer();

 // Configure and start the ADC timer
 configureADCtimer();

 chprintf(SERIAL," DSP Kernel started%s",BREAK);
 }

// Read potentiometer 1 on ADC3 as Q15 value
q15_t pot1read(void)
  {
  int i;
  int32_t value;

  // Set the multiplexer to pot1 input
  BIT_FIELD_WRITE(ADC3->SQR3,0,31,POT1_IN);

  value = 0;
  for(i=0;i<11;i++)
	 if (i>0) // Discard first reading
        {
        // Perform the reading
        ADC3->CR2 |= ADC_CR2_SWSTART;
        while (!(ADC3->SR & ADC_SR_EOC));
        value += (ADC3->DR);
        }

  return (1<<15) - ((value/10) << 3) - 1;
  }

// Read potentiometer 2 on ADC3 as Q15 value
q15_t pot2read(void)
  {
  int i;
  int32_t value;

  // Set the multiplexer to pot2 input
  BIT_FIELD_WRITE(ADC3->SQR3,0,31,POT2_IN);

  value = 0;
  for(i=0;i<11;i++)
	 if (i>0) // Discard first reading
        {
        // Perform the reading
        ADC3->CR2 |= ADC_CR2_SWSTART;
        while (!(ADC3->SR & ADC_SR_EOC));
        value += (ADC3->DR);
        }

  return (1<<15) - ((value/10) << 3) - 1;
  }

/** Timer Update RSI *****************************************/

// RSI for Timer8 update
// Implements the Single Sample DSP Kernel
CH_IRQ_HANDLER(VectorF0)
 {
 static int32_t count = 0;  // Counter for led blink
 static int32_t pos = 0;    // Counter for buffer position
 q15_t invalue1,outvalue1,invalue2,outvalue2;

 CH_IRQ_PROLOGUE();
 // Start of ISR code

 // Clear interrupt flag
 CLEAR_FLAG(TIM8->SR,TIM_SR_UIF);

 #ifdef BLOCK_MODE

 // Read ADC1 value from previous sample
 invalue1 = ((ADC1->DR)-2048)*16;

 // Read ADC2 value from previous sample
 invalue2 = ((ADC2->DR)-2048)*16;

 inbuff1[pos] = invalue1;  // Put value in input buffer 1
 inbuff2[pos] = invalue2;  // Put ADC2 value in input buffer 2

 // Check saturation
 if (invalue1 < SAT_LOW)  satCounter1 = SAT_COUNT;
 if (invalue1 > SAT_HIGH) satCounter1 = SAT_COUNT;
 if (invalue2 < SAT_LOW)  satCounter2 = SAT_COUNT;
 if (invalue2 > SAT_HIGH) satCounter2 = SAT_COUNT;

 // Check monitor user button
 if (USER_BUTTON_ACTIVE)
	 {
	 // Set output values from inputs
	 outvalue1 = invalue1;
	 outvalue2 = invalue2;
	 }
    else
     {
     // Get output values from buffers
     outvalue1 = outbuff1[pos];
     outvalue2 = outbuff2[pos];
     }

 // Set DAC1 value for next sample
 DAC->DHR12R1 = (outvalue1/16)+2048;

 // Set DAC2 value for next sample
 DAC->DHR12R2 = (outvalue2/16)+2048;

 // Update buffer position
 pos++;

 // Check half buffer
 if (pos == HBLOCK_SIZE)
     {
	 // Turn off GreenLED
	 GREEN_OFF;

	 // Set position
	 //inp1 = &(inbuff1[0]);
	 //inp2 = &(inbuff2[0]);
	 //outp1 = &(outbuff1[0]);
	 //outp2 = &(outbuff2[0]);
	 blkStart = 0;

	 // Signal the processing thread
	 chSysLockFromIsr();
	 chBSemSignalI(&blockSemaf);
	 chSysUnlockFromIsr();
     }

 // Check full buffer
 if (pos == (2*HBLOCK_SIZE))
     {
	 // Turn on GreenLED
	 GREEN_ON;

	 // Update position
	 pos = 0;

	 // Set position
	 //inp1 = &(inbuff1[HBLOCK_SIZE]);
	 //inp2 = &(inbuff2[HBLOCK_SIZE]);
	 //outp1 = &(outbuff1[HBLOCK_SIZE]);
	 //outp2 = &(outbuff2[HBLOCK_SIZE]);
	 blkStart = HBLOCK_SIZE;

	 // Signal the processing thread
	 chSysLockFromIsr();
	 chBSemSignalI(&blockSemaf);
	 chSysUnlockFromIsr();
     }

 #else // Code for sample to sample mode

 PRO1_SET; // Set profiling line #1

 // Blink Green LED at a frequency
 // 10000 times the sample frequency
 count++;

 if (count == 10000)
     {
     GREEN_ON;
     count = 0;
     }

 if (count == 5000)
     GREEN_OFF;

 // Read ADC1 value from previous sample
 invalue1 = ((ADC1->DR)-2048)*16;

 // Read ADC2 value from previous sample
 invalue2 = ((ADC2->DR)-2048)*16;

 // Check saturation
 if (invalue1 < SAT_LOW)  satCounter1 = SAT_COUNT;
 if (invalue1 > SAT_HIGH) satCounter1 = SAT_COUNT;
 if (invalue2 < SAT_LOW)  satCounter2 = SAT_COUNT;
 if (invalue2 > SAT_HIGH) satCounter2 = SAT_COUNT;

 // Perform the processing for this sample
 //PROC_FUNCTION(invalue1,invalue2,&outvalue1,&outvalue2);
 sampleProcessingPtr(invalue1,invalue2,&outvalue1,&outvalue2);

 // Check monitor user button
 if (USER_BUTTON_ACTIVE)
	 {
	 outvalue1 = invalue1;
	 outvalue2 = invalue2;
	 }

 // Set DAC1 value for next sample
 DAC->DHR12R1 = (outvalue1/16)+2048;

 // Set DAC2 value for next sample
 DAC->DHR12R2 = (outvalue2/16)+2048;

 PRO1_CLEAR; // Clear profiling line #1

 #endif // End of sample to sample code

 // End of the ISR code
 CH_IRQ_EPILOGUE();
 }


