/*********************************************************************
  B a s e . h

  Base header file for the TRS DSP board

  Version 1.1 (9/6/2015)

  Main include file elements:

    - References to HAL and OS includes
    - Bitfield definitions and pseudofunctions
    - ChibiOS pseudofunctions
    - System start function prototype

  Additional data:
    - MCU frequency is configured using the main PLL
      The mcuconf.h file includes the configuration information
      PLL input is HSE clock at 8MHz
         fvco=fpll_in*PLLN/PLLM = 8MHz*336/8=336MHz
         HCLK=fpll_out=fvco/PLLP = 336MHz/2=168MHz
         fusb_sdio_rng=fvco/PLLQ = 336MHz/7=48MHz
    - APB1 clock frequency depends on prescaler STM32_PPRE1
      Current value is STM32_PPRE1_DIV4 that corresponds to
         HCLK/4 = 168MHz / 4 = 42 MHz
    - APB2 clock frequency depends on prescaler STM32_PPRE2
      Current value is STM32_PPRE2_DIV2 that corresponds to
         HCLK/2 = 168MHz / 2 = 84 MHz
    - I2S clock frequency is generated on the PLLI2S PLL
      This PLL has the same input as the main PLL
         fvco=fpll_in*PLLI2SN/PLLM=8MHz*192/8=192MHz
         I2CCLK=fvco/PLLI2SR=192MHz/5=38.4MHz

*********************************************************************/

#ifndef BASE_H
#define BASE_H

// Other needed includes

#include "ch.h"             // Main ChibiOS include file
#include "hal.h"            // Main HAL include file
#include "DSP_Board_v11.h"  // DSP Board v1.1 definitions

// Current version
#define VERSION "1.0 (23/7/2019)"

/******************* Current MCU configuration **********************/

// Timer8 input clock frequency (in Hz)
#define TIM8_CLK_FREQ  168000000

/************************ Bit definitions ***************************/

// Generic bit definitions

#define BIT(n) (1<<(n))

// Bit definitions in a 32 bit word

#define BIT0   0x01
#define BIT1   0x02
#define BIT2   0x04
#define BIT3   0x08
#define BIT4   0x10
#define BIT5   0x20
#define BIT6   0x40
#define BIT7   0x80
#define BIT8   0x100
#define BIT9   0x200
#define BIT10  0x400
#define BIT11  0x800
#define BIT12  0x1000
#define BIT13  0x2000
#define BIT14  0x4000
#define BIT15  0x8000
#define BIT16  0x10000
#define BIT17  0x20000
#define BIT18  0x40000
#define BIT19  0x80000
#define BIT20  0x100000
#define BIT21  0x200000
#define BIT22  0x400000
#define BIT23  0x800000
#define BIT24  0x1000000
#define BIT25  0x2000000
#define BIT26  0x4000000
#define BIT27  0x8000000
#define BIT28  0x10000000
#define BIT29  0x20000000
#define BIT30  0x40000000
#define BIT31  0x80000000

/**** BIT MANIMPULATION PSEUDOFUNCTIONS ***************************/

// Check bit number bit of variable var
#define CHECK_BIT(var,bit)   ((var)&(1<<(bit)))

// Set in var bits set to "1" in flag
#define SET_FLAG(var,flag)     (var) = ((var)|(flag))

// Clear in var bits set to "1" in flag
#define CLEAR_FLAG(var,flag)   (var) = ((var)&(~(flag)))

// Return True if var has all flag bits set
#define CHECK_FLAG(var,flag)   (((var)&(flag)) == (flag))

// Set a in var, with value value, a bit field that starts on bit pos and has the indicated mask
#define BIT_FIELD_WRITE(var,pos,mask,value)  (var) = (((var)&(~((mask)<<(pos))))|((value)<<(pos)))

// Get the var bit field that starts on bit pos and has the indicated mask
#define BIT_FIELD_READ(var,pos,mask)   (((var)>>(pos))&(mask))

/**** Q15 related defines ***************************************/

#define Q15_ONE (1<<15)
#define Q15_MAX ((1<<15)-1)

/**** ChibiOS RELATED PSEUDOFUNCTIONS ***************************/

// ChibiOS blocking
#define DISABLE  chSysLock()
#define ENABLE   chSysUnlock()

// Sleep for the indicated number of milliseconds
// In a multithreaded application the sleep makes the
// CPU available to other threads
#define SLEEP_MS(x) chThdSleep(x)

/**** SERIAL DRIVER *****************************************/

// Definitions for serial
#define SERIAL (BaseSequentialStream*)&SD2  // Serial stream identifier
#define SERIALBASE (BaseChannel *)&SD2      // Serial base channel identifier
#define BREAK "\r\n"                        // Line break
#define SSPEED 115200   // Serial speed

// Serial driver used
// It shall be configured on halconf.h and mcuconf.h
#define SDRIVER SD2

/********************* FUNCTION PROTOTYPES ******************/

/// Main initializations
// It has to be called at the start of main
void baseInit(void);

#endif //BASE_H



