/*
   main.c
   Main C source file
   Test for SD SPI Module
 */

#include "Base.h"           // Basic definitions
#include "chprintf.h"       // ChibiOS/RT chprintf function
#include "uart6_tx.h"       // Serial debug header file
#include "ff.h"             // Fat File System header file
#include "sd_spi.h"         // Card Driver header file

// Example short text file in the card
#define SHORT_FILE "\\test.txt"

// Example big file in the card (Some MB)
#define LONG_FILE "\\Discovery.pdf"

// Name for the big file to be created
#define LONG_CREATE "\\Created File.bin"

// MMC Driver
extern MMCDriver MMCD1;

// FAT File System Object
extern FATFS MMC_FS;

// MMC Card Block Device information
BlockDeviceInfo mmcBdip;

// 1KB Buffer for one Card Block
uint8_t Buffer[1024];

/************************* GENERIC STATIC FUNCTIONS ******************************/

// Stop the program as it cannot continue
// This function never returns
static void panicAbort(void)
 {
 chprintf(SDEBUG,"Aborting program and lock in a loop%s",BREAK);
 chprintf(SDEBUG,"BYE%s",BREAK);
 while (1) {};
 }

/************************* MMC LOW LEVEL STATIC FUNCTIONS ******************************/

// Test the Card Detect and Write protect Switches
// Green Led will be ON when card is detected
// Red Led will be ON when card is write protected
// If no card is inserted it reads as Write Protected
//
static void testSDswitches(void)
 {
 while(1) // Infinite loop
    {
	// Check card detect switch
	if (sdIsDetected())
		(LEDS_PORT->BSRR.H.set)=BIT(GREEN_LED_PAD);
	    else
	    (LEDS_PORT->BSRR.H.clear)=BIT(GREEN_LED_PAD);

	// Check Write Protect switch
	if (sdIsWriteProtected())
		(LEDS_PORT->BSRR.H.set)=BIT(RED_LED_PAD);
		else
		(LEDS_PORT->BSRR.H.clear)=BIT(RED_LED_PAD);

	// Sleep 0.2s
	SLEEP_MS(200);
    }

 }

// Show Card Information
static void showCardInfo(void)
 {
 int32_t size;

 if (mmcGetInfo(&MMCD1,&mmcBdip)==CH_FAILED)
       {
	   chprintf(SDEBUG,"Cannot get MMC Card information%s",BREAK);

	   // Stop program execution
	   panicAbort();
       }

 size=(mmcBdip.blk_size*mmcBdip.blk_num)/1024/1024;

 chprintf(SDEBUG,"Card Device information:%s",BREAK);
 chprintf(SDEBUG,"  Block size: %d Bytes%s",mmcBdip.blk_size,BREAK);
 chprintf(SDEBUG,"  Number of blocks: %d%s",mmcBdip.blk_num,BREAK);
 chprintf(SDEBUG,"  Total card size: %d MBytes%s",size,BREAK);
 }

// Low level dump of blocks
// Dumps information on screen
// Returns 0 if OK
int32_t sdDumpBlocks(int32_t first, int32_t count)
  {
  int i,line,lines,pos;

  // Number of 16 Bytes lines in a block
  lines=mmcBdip.blk_size/16;

  // Check first block number
  if ((first<0)||(first>=((int32_t)mmcBdip.blk_num)))
        {
	    chprintf(SDEBUG,"Block number %d is out of limits%s",first,BREAK);
        return 1;
        }

  // Check block count
  if ((first+count-1)>=((int32_t)mmcBdip.blk_num))
         {
 	     chprintf(SDEBUG,"Block count %d is out of limits%s",first,BREAK);
         return 2;
         }

  // Start a low level block sequential read
  if (mmcStartSequentialRead(&MMCD1,first)==CH_FAILED)
       {
       chprintf(SDEBUG,"Cannot start a sequential block read%s",BREAK);

       // Stop program execution
       panicAbort();
       }

  // Read the selected number of blocks
  for(i=0;i<count;i++)
     {
	 // Show Block header
	 chprintf(SDEBUG,"%sBlock number %d%s",BREAK,first+i,BREAK);

	 // Read this block
     if (mmcSequentialRead(&MMCD1,Buffer)==CH_FAILED)
          {
    	  chprintf(SDEBUG,"Cannot read card block %d%s",first+i,BREAK);

    	  // Stop program execution
    	  panicAbort();
          }

     // Show block contents
     for(line=0;line<lines;line++)
         {
    	 chprintf(SDEBUG,"%5d: ",line*16);
    	 for(pos=0;pos<16;pos++)
    		 chprintf(SDEBUG,"%4d",Buffer[line*16+pos]);
    	 chprintf(SDEBUG,"%s",BREAK);
         }
     }

  // Stop a low level block sequential read
  if (mmcStopSequentialRead(&MMCD1)==CH_FAILED)
       {
       chprintf(SDEBUG,"Error at stop of the sequential block read%s",BREAK);

       // Stop program execution
       panicAbort();
       }

  // Returns OK
  return 0;
  }

// Give some information obtained from block 0
// This supposes that the volume if FAT
// Information obtained from:
// http://www.tavi.co.uk/phobos/fat.html
void sdBlock0info(void)
 {
 int32_t blkSize;

 // Start a low level block sequential read at block 0
 if (mmcStartSequentialRead(&MMCD1,0)==CH_FAILED)
	 {
	 chprintf(SDEBUG,"Cannot start a sequential block read at block 0%s",BREAK);

	 // Stop program execution
	 panicAbort();
	 }

 // Read Block 0
 if (mmcSequentialRead(&MMCD1,Buffer)==CH_FAILED)
     {
     chprintf(SDEBUG,"Cannot read card block 0%s",BREAK);

     // Stop program execution
	 panicAbort();
	 }

 // Show information
 chprintf(SDEBUG,"%sBasic FAT information from block 0%s",BREAK,BREAK);
 // Check block 0 signature
 if ((Buffer[510]==0x55)&&(Buffer[511]==0xAA))
	 chprintf(SDEBUG,"  Block 0 signature 0x55 0xAA is correct%s",BREAK);
    else
     {
     chprintf(SDEBUG,"Bad block 0 signature%s",BREAK);

     // Stop program execution
     panicAbort();
     }

 // Number of bytes per block at positions at 11(LSB) and 12(MSB)
 blkSize=Buffer[11]+Buffer[12]*256;
 chprintf(SDEBUG,"  Bytes per block: %d%s",blkSize,BREAK);
 // Number of block per allocation unit at 13
 chprintf(SDEBUG,"  Number of blocks per allocation unit: %d (%d Bytes)%s"
		 ,Buffer[13],Buffer[13]*blkSize,BREAK);
 // Number of reserved blocks at 14
 chprintf(SDEBUG,"  Number of reserved blocks: %d%s",Buffer[14],BREAK);

 // Stop a low level block sequential read
 if (mmcStopSequentialRead(&MMCD1)==CH_FAILED)
      {
      chprintf(SDEBUG,"Error at stop of the sequential block read%s",BREAK);

      // Stop program execution
      panicAbort();
      }
 }

// Calculates the time needed to read at low level
// Supposes that CH_FREQUENCY is 1000 as defined in chconf.h
//
// As SPI works at 18MHz in this program the throughput
// cannot be better that 18Mbits/s or 2.25 MBytes/s
void timeLowLevelRead(void)
 {
 int32_t start,elapsed,count;

 chprintf(SDEBUG,"%sLow level 1000 sector read test...%s",BREAK,BREAK);

 // Start a low level block sequential read at block 0
  if (mmcStartSequentialRead(&MMCD1,0)==CH_FAILED)
 	 {
 	 chprintf(SDEBUG,"Cannot start a sequential block read at block 0%s",BREAK);

 	 // Stop program execution
 	 panicAbort();
 	 }

  // Get time of start
  start=chTimeNow();

  // Read the selected number of blocks
  count=1000;
  do
     {
 	 // Read one block
     if (mmcSequentialRead(&MMCD1,Buffer)==CH_FAILED)
           {
     	  chprintf(SDEBUG,"Cannot read card block%s",BREAK);

     	  // Stop program execution
     	  panicAbort();
           }
     }
     while (--count);


 // Get time of stop
 elapsed=chTimeElapsedSince(start);

 // Stop a low level block sequential read
 if (mmcStopSequentialRead(&MMCD1)==CH_FAILED)
      {
      chprintf(SDEBUG,"Error at stop of the sequential block read%s",BREAK);

      // Stop program execution
      panicAbort();
      }

 chprintf(SDEBUG,"  Time to read 1000 blocks: %d ms%s",elapsed,BREAK);
 chprintf(SDEBUG,"  Throughput of %d Bytes/s%s",(mmcBdip.blk_size*1000*1000)/elapsed,BREAK);
 }

/************************* HIGH LEVEL FAT FS STATIC FUNCTIONS ***********************/

// Test reading the file SHORT_FILE
static void testReadFile(void)
 {
 FIL File;
 int32_t size,i;
 UINT nread;

 // Try to open the file
 if (f_open(&File,SHORT_FILE,FA_READ)!=FR_OK)
     {
	 mmcDisconnect(&MMCD1);
	 chprintf(SDEBUG,"Cannot open the file %s%s",SHORT_FILE,BREAK);

	 // Stop program execution
	 panicAbort();
     }

 chprintf(SDEBUG,"File %s opened%s",SHORT_FILE,BREAK);

 // Get file size
 size=f_size(&File);
 chprintf(SDEBUG,"Length of file is %d bytes%s",size,BREAK);

 // Try to read the file
 if (f_read(&File,Buffer,size,&nread)!=FR_OK)
     {
 	 mmcDisconnect(&MMCD1);
 	 chprintf(SDEBUG,"Cannot read the file%s",BREAK);

 	 // Stop program execution
 	 panicAbort();
     }

 // Show file contents
 chprintf(SDEBUG,"File contents:%s%s",BREAK,BREAK);
 for(i=0;i<size;i++)
	 chnPutTimeout(BASEDEBUG,Buffer[i],TIME_INFINITE);

 // Separate from following lines
 chprintf(SDEBUG,"%s%s",BREAK,BREAK);

 // Try to close the file
 if (f_close(&File)!=FR_OK)
     {
 	 mmcDisconnect(&MMCD1);
 	 chprintf(SDEBUG,"Cannot close the file%s",BREAK);

 	 // Stop program execution
 	 panicAbort();
     }

 chprintf(SDEBUG,"File %s closed%s",SHORT_FILE,BREAK);
 }

// Test speed reading the file LONG_FILE
static void timeFATread(void)
 {
 FIL File;
 int32_t size,blocks;
 UINT nread;
 int32_t start,elapsed;

 chprintf(SDEBUG,"%sFile read throughput test%s",BREAK,BREAK);

 // Try to open the file
 if (f_open(&File,LONG_FILE,FA_READ)!=FR_OK)
     {
	 mmcDisconnect(&MMCD1);
	 chprintf(SDEBUG,"Cannot open the file %s%s",LONG_FILE,BREAK);

	 // Stop program execution
	 panicAbort();
     }

 chprintf(SDEBUG,"File %s opened%s",LONG_FILE,BREAK);

 // Get file size
 size=f_size(&File);
 chprintf(SDEBUG,"Length of file is %d bytes%s",size,BREAK);
 chprintf(SDEBUG,"Reading file. This can take some time...%s",BREAK);

 // No blocks read yet
 blocks=0;

 // Get time of start
 start=chTimeNow();

 do
  {
  if (f_read(&File,Buffer,1024,&nread)!=FR_OK)
	     {
	 	 mmcDisconnect(&MMCD1);
	 	 chprintf(SDEBUG,"Cannot read the file%s",BREAK);

	 	 // Stop program execution
	 	 panicAbort();
	     }
  blocks++;
  }
  while(nread==1024);

 // Get time of stop
 elapsed=chTimeElapsedSince(start);

 chprintf(SDEBUG,"File read using %d 1K Blocks%s",blocks,BREAK);
 chprintf(SDEBUG,"Time to read the file is %d ms%s",elapsed,BREAK);
 chprintf(SDEBUG,"Throughput is %d Bytes/s%s"
		  ,(int32_t)((((int64_t)1000)*(int64_t)size)/elapsed),BREAK);

 // Try to close the file
 if (f_close(&File)!=FR_OK)
     {
 	 mmcDisconnect(&MMCD1);
 	 chprintf(SDEBUG,"Cannot close the file%s",BREAK);

 	 // Stop program execution
 	 panicAbort();
     }

 chprintf(SDEBUG,"File %s closed%s",LONG_FILE,BREAK);
 }

// Test write speed on LONG_CREATE file
static void timeFATwrite(void)
 {
 FIL File;
 int32_t blocks,i;
 UINT nwrite;
 int32_t start,elapsed;

 chprintf(SDEBUG,"%sFile write throughput test%s",BREAK,BREAK);

 // Try to open the file
 if (f_open(&File,LONG_CREATE,FA_WRITE|FA_CREATE_ALWAYS)!=FR_OK)
     {
	 mmcDisconnect(&MMCD1);
	 chprintf(SDEBUG,"Cannot create the file %s%s",LONG_CREATE,BREAK);

	 // Stop program execution
	 panicAbort();
     }

 chprintf(SDEBUG,"File %s created%s",LONG_CREATE,BREAK);

 // Fill the buffer
 for(i=0;i<1024;i++)
	 Buffer[i]=(uint8_t)i;

 chprintf(SDEBUG,"Writing 200KB file in 1K Blocks. This can take some time...%s",BREAK);

 // Get time of start
 start=chTimeNow();

 // We will write 200 1K blocks
 blocks=200;
 do
  {
  if (f_write(&File,Buffer,1024,&nwrite)!=FR_OK)
	     {
	 	 mmcDisconnect(&MMCD1);
	 	 chprintf(SDEBUG,"Cannot write to the file%s",BREAK);

	 	 // Stop program execution
	 	 panicAbort();
	     }
  blocks--;
  }
  while(blocks);

 // Get time of stop
 elapsed=chTimeElapsedSince(start);

 chprintf(SDEBUG,"File write using 200 1K Blocks%s",blocks,BREAK);
 chprintf(SDEBUG,"Time to write the file is %d ms%s",elapsed,BREAK);
 chprintf(SDEBUG,"Throughput is %d Bytes/s%s"
		  ,(200*1024*1000)/elapsed,BREAK);

 // Try to close the file
 if (f_close(&File)!=FR_OK)
     {
 	 mmcDisconnect(&MMCD1);
 	 chprintf(SDEBUG,"Cannot close the file%s",BREAK);

 	 // Stop program execution
 	 panicAbort();
     }

 chprintf(SDEBUG,"File %s closed%s",LONG_FILE,BREAK);
 }



/************************* MAIN FUNCTION ******************************/

int main(void)
 {
 // Global initializations
 baseInit();

 // Start the serial channel 6 TX
 uart6start();

 // Initialize the Card Driver
 sdDriverInit();

 // Test the Card Detect and Write protect switches
 // This function never returns
 //testSDswitches();

 // Try to detect if the card is present
 if (sdIsDetected())
      {
	  chprintf(SDEBUG,"SD Card Detected ");
	  if (sdIsWriteProtected())
	      {
		  chprintf(SDEBUG,"in read only mode%s",BREAK);
		  // Turn on red led
		  (LEDS_PORT->BSRR.H.set)=BIT(RED_LED_PAD);
	      }
	     else
	      chprintf(SDEBUG,"in read/write mode%s",BREAK);

	  // Turn on green led
	  (LEDS_PORT->BSRR.H.set)=BIT(GREEN_LED_PAD);
      }
     else
     {
     chprintf(SDEBUG,"SD Card not detected%s",BREAK);

     // Stop program execution
     panicAbort();
     }

 // Start low level tests
 chprintf(SDEBUG,"%sLow Level MMC tests -----------------%s",BREAK,BREAK);

 // Connect to SD Card

 if (mmcConnect(&MMCD1)==CH_FAILED)
   {
   chprintf(SDEBUG,"Cannot connect to MMC Card using SPI mode%s",BREAK);

   // Stop program execution
   panicAbort();
   }
 chprintf(SDEBUG,"Connected to MMC Card%s",BREAK);

 // Show card information
 showCardInfo();

 // Dump three card blocks
 chprintf(SDEBUG,"%sTest to dump Card Blocks 0 to 2%s",BREAK,BREAK);
 sdDumpBlocks(0,3);

 // Some information on block 0
 sdBlock0info();

 // Time low level read operation
 timeLowLevelRead();

 // Start high level Fat tests
 chprintf(SDEBUG,"%sHigh Level Fat File System tests --------------%s",BREAK,BREAK);

 // Mount the File System
 if(f_mount(0, &MMC_FS)!=FR_OK)
    {
	mmcDisconnect(&MMCD1);
	chprintf(SDEBUG,"Cannot mount the FAT File System%s",BREAK);

	// Stop program execution
	panicAbort();
    }

 chprintf(SDEBUG,"FAT File System mounted%s",BREAK);

 // Read a short file
 testReadFile();

 // Time the read operation on a big file
 timeFATread();

 // Time the write operation on a big file
 timeFATwrite();

 // Disconnect from MMC Card
 if (mmcDisconnect(&MMCD1)==CH_FAILED)
    {
    chprintf(SDEBUG,"Cannot disconnect from MMC Card%s",BREAK);

    // Stop program execution
    panicAbort();
    }
 chprintf(SDEBUG,"Disconnected from MMC Card%s",BREAK);

 // Infinite loop
 chprintf(SDEBUG,"%sProgram End. Lock in a loop.%s",BREAK,BREAK);
 while (1) {};

 // Main should never return
 // This line is included so that the compiler doesn't complain
 return 0;
 }

