/*
   main.c
   Main C source file
   Test for ccm_debug

   This project needs memstreams.c and chprintf.c
   to be included in the makefile

   Using the default 64 (0x40) char line length there are 1024 lines
   First line starts at 0x10000000
   Last line starts at  0x1000FFC0

   Information before 0x10000000 and after 0x10010000 is undefined (reserved)
   so they may contain alias of another regions of memory or any other crap.

   To view the CCM contents in text mode we need to have a "Memory" view
   in the Debug perspective. Works best as a tab next to Variables and Expressions
   We will need to configure from 0x10000000 in ASCII mode with
   64 bytes per line and 64 bytes per column.
 */

// We activate debug over CCM
// If we don't set this variable, all uppercase CCM commands
// will be eliminated by the preprocessor
// That way you can eliminate all debug operations on the
// final release program
#define ACTIVATE_CCMDEBUG

#include "Base.h"           // Basic definitions
#include "chprintf.h"       // ChibiOS/RT chprintf function
#include "ccm_debug.h"      // CCM Debug header file

// Calculate time needed for CCM operations on first fill
// Supposes that CH_FREQUENCY is 1000 as defined in chconf.h
void timeCCM(int32_t clear)
 {
 int32_t start,elapsed,i;

 if (clear) CCMCLEAR(); // Clear CCM

 // Get time of start
 start=chTimeNow();

 // 1024 50 char line test
 for(i=0;i<1024;i++)
	 CCMPRINTF("01234567890123456789012345678901234567890123456789");

 // Get time of stop
 elapsed=chTimeElapsedSince(start);

 CCMPRINTF("Time to write 1024 lines is %d ms",elapsed);
 CCMPRINTF("Throughput is %d lines/s",(1024*1000)/elapsed);
 CCMPRINTF("Or also %d char/s",50*((1024*1000)/elapsed));
 CCMPRINTF("This is equivalent to %d baud RS-232",100*50*((1024*1000)/(8*elapsed)));
 }

int main(void)
 {
 int i;

 // Global initializations
 baseInit();

 // Clear the CCM Memory
 // We will do that always regardless of ACTIVATE_CCMDEBUG variable
 // The rest of CCM operations will be carried out only if the
 // ACTIVATE_CCMDEBUG has been defined
 ccmClear();

 // Write a message to CCM
 CCMPRINTF("This is a CCM test");
 CCMPRINTF("This is a line with a %d number",10);
 CCMPRINTF("This is very long line that will be truncated as maximum size is 64 characters");

 // 1024 line test to demonstrate wrapping
 for(i=0;i<1024;i++)
	 CCMPRINTF("This is test line number %d",i);

 // Time the CCM operation clearing the memory
 timeCCM(1);

 // Time the CCM operation without clearing the memory
 // Throughput will be less as each line needs to be cleared before writing
 //timeCCM(0);

 // Infinite loop
 while (1) {};

 // Main should never return
 // This line is included so that the compiler doesn't complain
 return 0;
 }

