/*
 uart6_tx.c
 Use of the CCM Memory for debugging messages
 */

#include "Base.h"           // Basic definitions
#include "chprintf.h"	    // CHIBIOS/RT Printf module
#include "memstreams.h"     // CHIBIOS/RT Memory stream module
#include "ccm_debug.h"      // This module header file

/******************* INTERNAL VARIABLES **********************/

// CCM Position (we start at zero)
int32_t ccmPosition=0;

// CCM Wrapped from the end
// If true if all the CCM memory has been used and we
// restart from the beginning of CCM
int32_t ccmWrapped=0;

/******************* PUBLIC VARIABLES ************************/

// CCM Pointer (we start at CCM start position)
char (*ccmPointer)=CCM_CHAR;

/****************** STATIC FUNCTIONS ************************/

// Writes the EOF indication
static inline void ccmWriteEOF(void)
 {
 int32_t i;

 // We clear the line
 //for(i=0;i<CCM_LINESIZE;i++)
//	          ccmPointer[i]=0;

 // If wrapped, we need to clear the line
 i=CCM_LINESIZE;
 do
  {
  ccmPointer[--i]=0;
  }
  while (i);

 // We write EOF
 chsnprintf(ccmPointer,CCM_LINESIZE,"-----<EOF>-----");
 }

/****************** PUBLIC FUNCTIONS ************************/

// Clear CCM Memory
void ccmClear(void)
 {
 int32_t i;
 int32_t *pointer;

 // Erase all CCM positions to zero
 i=CCM_SIZE/sizeof(int32_t);
 pointer=CCM_INT;
 do
  {
  (*pointer)=0;
  pointer++;
  }
  while (--i);

 // Reset CCM Position
 ccmPosition=0;

 // Reset CCM Pointer
 ccmPointer=CCM_CHAR;

 // Reset the wrapping variable
 ccmWrapped=0;
 }

// Increase line pointer
void ccmLine(void)
 {

 // Increase position
 ccmPosition+=CCM_LINESIZE;

 // Check out of bounds
 if (ccmPosition>=CCM_SIZE)
     {
	 ccmPosition=0;  // Restart at zero
	 ccmWrapped=1;   // Indicate the wrapping effect
     }

 // Set pointer
 ccmPointer=CCM_CHAR+ccmPosition;

 // Set EOF marker if we have wrapped
 if (ccmWrapped) ccmWriteEOF();
 }

// Write a message to CCM memory
void ccmWrite(char *cad)
 {
 int i;

 // If wrapped, we need to clear the line
 i=CCM_LINESIZE;
 do
  {
  ccmPointer[--i]=0;
  }
  while (i);


 // Check if line is empty
 if (!(*cad))
     {
	 ccmLine();  // Jump to next line
	 return;
     }

 // Write the line
 i=0;
 do
   {
   ccmPointer[i++]=(*cad);

   cad++; // Increase pointer
   }
   while ((*cad)&&(i<CCM_LINESIZE));

 // Jump to next line
 ccmLine();
 }


